#pragma once
#ifndef __MAIN_H
#define __MAIN_H

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <assert.h>
#include <string.h>
#include <time.h>

// we don't need ALL windows featues
#define WIN32_LEAN_AND_MEAN
#include <Windows.h>

// A structure used for ship placement
typedef struct ShipsToPlace
{
	int length,count;
	struct ShipsToPlace *next;
} ShipsToPlace;

// We're writing to the raw memory buffer of the windows command line
// window. This structure keeps track of all the necessary data,
// as well as some variables for the tiny console at the bottom
// of the window.
typedef struct WinConsole
{
	HANDLE wHnd;    // Handle to write to the console.
	HANDLE rHnd;    // Handle to read from the console.
   SMALL_RECT windowSize;
   COORD bufferSize;
   CHAR_INFO buf[80*25]; // the character buffer for the console
   COORD charBufSize;
   COORD characterPos;
   SMALL_RECT writeArea;

	int updating; // whether the screen is currently updating or not
	int consoleStart; // the y-position at which the tiny console starts
	int consoleX, consoleY; // the current cursor position
	int consoleEnd; // the y-position at which the tiny console ends
	COORD cursorPos;
} WinConsole;

// structure representing a game. It holds
// all the game variables, and the AI state
typedef struct Game
{
	// We use an array of bytes to represent the board/map. Each byte
	// represents one grid position. The first value is always the
	// top-left corner (A1). The game is setting and getting
	// flags on each byte to keep track of what is happening
	unsigned char *mapPlayer, *mapOpponent;
	ShipsToPlace *shipsToPlace;
	int playerWon, opponentWon;
	WinConsole con;
	// data used by tracking AIs
	int aiHitx;
	int aiHity;
	int aiState; // 0 - shooting at random, 1 - tracking a ship
	int aiLuckyShots; // only used by the cheating AI.
} Game;

// bit fields for board:
// 1 - ship present here
// 2 - shot fired here
// 4 - ship neighbouring here
// 8 - sunk
// 16 - neighboring field was shot (Some AIs use this)
// 32 - don't care
// 64 - don't care
// 128 - don't care
#define F_SHIP_PRESENT 1
#define F_SHOT_FIRED 2
#define F_NEIGHBOURING 4
#define F_SUNK 8
#define F_NEIGHBOR_SHOT 16

// some quick macros for convenience
#define XYVALID(x,y) (x >= 0 && x < gameWidth && y >= 0 && y < gameHeight)
#define XVALID(x) (x  >= 0 && x < gameWidth)
#define YVALID(y) (y >= 0 && y < gameHeight)

// used for placing ships. Is it placed horizontally or vertically?
typedef enum Orientation {horizontal, vertical } Orientation;
// used for printing the map, it influences what information is shown
typedef enum View {player, opponent, playerPlacement, debug } View;
// player - show everything
// opponent - only show the fields that were hit
// playerPlacement - just like player, but also show margins around the ships
// debug - Show raw byte values represented in a base-26 system (A = 0, B = 1, ...)

// a helper macro for the windows console
#define ATTS_DEFAULT (FOREGROUND_RED | FOREGROUND_GREEN | FOREGROUND_BLUE)

// ******************************************************************
// **								AI OPPONENT										 **
// **																					 **
// ******************************************************************

// a helper function for AI shooting
int AI_Shoot(unsigned char* map, int x, int y, Game *g);

// a function that picks a random position for shooting.
// the position of the shot is written to x and y
// if useGrid is set to 1, the algorithm will use the tracking grid
// if avoidNeighbouring is set to 1, the algorithm will not fire at spots
// that are neighbouring already hit/sunk ships.
int shootAtRandom(Game *g, int *x, int *y, int useGrid, int avoidNeighbouring);

// The stupid AI shoots completely at random
// It doesn't even care if it hits anything
void stupidAITurn(Game *g);

// The Adept AI shoots at random just like the stupid AI,
// but it can track its target once it hits someting
void adeptAITurn(Game *g);

// the Expert AI uses the grid strategy
// to locate your ships faster
void ExpertAITurn(Game *g);

// The most challenging AI will use
// the grid strategy, as well as a limited possibility
// of looking up your ships. It can be adjusted
// by cheating aggressiveness factor (the higher it is
// the more often the AI cheats.)
// cheating is decided on per-shot basis. Whenever a
// cheating shot occurs, the game comments on 
// how 'lucky' the shot was. There is a fixed limit
// on how many times the AI can cheat per game.
void CheatingAITurn(Game *g);

// a function pointer that is set the chosen AI function
// (One of the *AITurn functions above)
void (*doOpponentsTurn)(Game*);

// ******************************************************************
// **								AI TRACKING										 **
// **																					 **
// ******************************************************************
// AI tracking means that the AI has managed to discover a position
// of one of your ships, and it attempts sink it.

// this function is called whenever the AI is in tracking mode,
// and a cheating shot is generated. It looks up where your ship is.
int trackTargetCheating(Game *g);

// The standard, non-cheating target tracking function.
int trackTarget(Game *g);

// A helper function for tracking.
int TrackingAILoop(Game *g, int useGrid, int useCheating);

// a base function for tracking AIs.
void TrackingAIBase (Game *g, int useGrid, int useCheating);

// ******************************************************************
// **								CONSOLE WIN										 **
// **																					 **
// ******************************************************************
// Turns out that writing to the raw windows console is not simple.

// Initialize WinConsole structure
// TODO: Change parameter to WincConsole* type?
void InitWinConsole(Game *g);

// Copy the character buffer to the screen
void WinConsoleUpdate(WinConsole *c);

// begin updating the console. After this is called, any function
// that operates on the character buffer (for example printfCon)
// will NOT copy the data to the screen. This speeds up some functions
// significantly.
void WinConsoleBegin(WinConsole *c);

// End updating the console. The idea is that we place any opearations
// that heavily alter the character buffer between a
// WinConsoleBegin() and WinConsoleEnd() pair.
void WinConsoleEnd(WinConsole *c);

// A helper function for getting the character buffer data at (x,y)
CHAR_INFO *GetConBuf(WinConsole *wc, int x, int y);

// This function is like standard printf, but it prints to the
// raw character buffer at the given location, and with the given
// attributes. Windows uses attributes to define extra character info
// like for example background color.
void printfxy(Game *g, int x, int y, int atts, const char *format, ...);

// Match the Windows console cursor position with the one defined
// in the WinConsole structure.
void UpdateWinCursor(WinConsole *wc);

// This function scrolls the tiny console at the bottom of the application.
void scrollWinConsole(WinConsole *wc);

// Since we are using raw console access, there is no echo
// after the user presses Enter. We have print the echo
// manually. This helper function does that while also
// reading the input from the user.
void consoleInputWin(WinConsole *con, char *buf, int buflen);

// Prints characters to the tiny console, using the position
// variables in the WinConsole structure
void printfCon(WinConsole *wc, const char *format, ...);

// a helper function for printMapWin. It prints the chosen map
// at the give location on the screen, and using specified View.
void printMapWinInternal(Game *g, unsigned char *map, int ox, int oy, View v);

// Prints both game maps side-by-side using the Windows Console
// functionality
void printMapWin(Game *g, View playerView, View opponentView);

// ******************************************************************
// **								GAME												 **
// **																					 **
// ******************************************************************

// initialize the Game structure
void gameInit(Game *g);

// Cleanup the Game structure (free memory)
void gameClose(Game *g);

// Has the player won?
int testVictoryCondition(Game *g);

// Has the AI won?
int testDefeatCondition(Game *g);

// Let the player fire their shots
void doPlayersTurn(Game *g);

// The main loop. Once called it keeps going until
// the game is over.
void MainLoop(Game *g);

// ******************************************************************
// **								MAP												 **
// **																					 **
// ******************************************************************

// A helper function that translates Map Coordinates to grid number.
// For example (0,0) -> "A1", (1,0) -> "B1", (10,10) -> "J10" etc.
void MapCoordsToGrid(int x, int y, char *outbuf, int buflen);

// a helper function that gets the map cell byte value at given map coordinates
int mapGet(unsigned char* map, int x, int y);

// a helper function that sets the map cell byte value at given map coordinates
void mapSet(unsigned char* map, int x, int y, unsigned char value);

// A set of helper functions that Set, Clear, or check for presence of a give Flag
// at given map coordinates. The Flag can be one of the F_* macros.
void mapSetF(unsigned char* map, int x, int y, unsigned char flag);
void mapClearF(unsigned char* map, int x, int y, unsigned char flag);
int mapCheckF(unsigned char* map, int x, int y, unsigned char flag);

// An old function that prints the chosen map witout using the Windows Console
// functionality. Not used anymore.
void printMap (unsigned char* map, View v); // DEPRECEATED

// Are there any non-sunk ships left on the map?
int areThereAnyShipsSurviving(unsigned char *map);

// returns 0 if miss, 1 if hit, 2 if sunk, and -1 if
// the target was hit already (which may or may not be an error)
// This function does the actual shooting
int Shoot(unsigned char* map, int x, int y, WinConsole *con);

// ******************************************************************
// **								MAP												 **
// **																					 **
// ******************************************************************

// Initialize a ShipsToPlace structure
void shipsToPlaceInit(Game *g);

// Cleanup a ShipsToPlace structure (Free memory)
void shipsToPlaceClose(Game *g);

// Place a ship on the map at given coordinates and orientation.
// Return 1 on success, 0 on failure
int placeShip(unsigned char *map, int shipSize, int px, int py, Orientation ori);

// Determines whether all available ships have been placed
int areThereAnyMoreShipsLeftToPlace(Game *g);

// Prints the "You have this many ships of this kind left to place"
// messages
void printNumberOfShipsToPlace(Game *g);

// This function parses the ships placement command given by the player.
// For example: "A1 3 v" means "place a three-master on A1 vertically"
int parseShipPlacementCommand(Game *g, char *cmd, int cmdlen);

// Does ship placement automatically, can also be used for player ships.
void placeOpponentsShips(Game *g, unsigned char *map);

// Place ships manually, asking the player for positions.
void placePlayersShips(Game *g);

// ******************************************************************
// **								UTIL												 **
// **																					 **
// ******************************************************************

// WARNING: It modifies the parameter.
// Converts a given string to lowercase ("ABC10" -> "abc10")
char *stringToLower(char *str, int maxlen);

// Converts a string containing a number to int
int parseNumber(char *str, int len, int *res);

// prereq: cmd must be lowercased!
// Converts grid location to map coordinates ("A1" -> (0,0))
int parseGridSquare(char *cmd, int cmdlen, int *resx, int *resy);

// Math.h lacks power function for integers
int pow (int x, int power) ;

// Math.h lacks abs for integers
int abs (int val);

// ******************************************************************
// **								MAIN												 **
// **																					 **
// ******************************************************************

// These constants are set in Main.c and can be used
// to alter the parameters of the game.
const int numSizeFiveShips;
const int numSizeFourShips;
const int numSizeThreeShips;
const int numSizeTwoShips;
const int maxShipLen;
const int shotsPerTurn;
const int gameWidth;
const int gameHeight;
const int numAiLuckyShots;

#endif